<?php
/**
 * Copyright 2002-2016 Horde LLC (http://www.horde.org/)
 *
 * See the enclosed file COPYING for license information (LGPL). If you
 * did not receive this file, see http://www.horde.org/licenses/lgpl21.
 *
 * @author    Chuck Hagenbuch <chuck@horde.org>
 * @author    Michael J. Rubinsky <mrubinsk@horde.org>
 * @author    Jan Schneider <jan@horde.org>
 * @category  Horde
 * @license   http://www.horde.org/licenses/lgpl21 LGPL-2.1
 * @package   Image
 */

/**
 * The Horde_Image_Effect parent class defines a general API for ways to apply
 * effects to Horde_Image objects.
 *
 * @author    Chuck Hagenbuch <chuck@horde.org>
 * @author    Michael J. Rubinsky <mrubinsk@horde.org>
 * @author    Jan Schneider <jan@horde.org>
 * @category  Horde
 * @copyright 2002-2016 Horde LLC
 * @license   http://www.horde.org/licenses/lgpl21 LGPL-2.1
 * @package   Image
 */
class Horde_Image_Effect
{
    /**
     * Effect parameters.
     *
     * @var array
     */
    protected $_params = array();

    /**
     * The bound Horde_Image object
     *
     * @var Horde_Image
     */
    protected $_image = null;

    /**
     * Logger.
     */
    protected $_logger;

    /**
     * Effect constructor.
     *
     * @param array $params  Any parameters for the effect. Parameters are
     *                       documented in each subclass.
     */
    public function __construct($params = array())
    {
        $this->_logger = new Horde_Support_Stub();
        foreach ($params as $key => $val) {
            $this->_params[$key] = $val;
        }
    }

    /**
     * Bind this effect to a Horde_Image object.
     *
     * @param Horde_Image $image  The Horde_Image object
     */
    public function setImageObject($image)
    {
        $this->_image = $image;
    }

    /**
     * Attaches a logger.
     *
     * @param Horde_Log_Logger $logger  A logger.
     */
    public function setLogger($logger)
    {
        $this->_logger = $logger;
    }

    /**
     * Effect factory.
     *
     * @param string $type    An effect name.
     * @param string $driver  An image driver name.
     * @param array $params   Any effect parameters.
     *
     * @return Horde_Image_Effect  An effect instance.
     */
    public static function factory($type, $driver, $params)
    {
        if (is_array($type)) {
            list($app, $type) = $type;
        }

        // First check for a driver specific effect, if we can't find one,
        // assume there is a vanilla effect object around.
        $class = 'Horde_Image_Effect_' . $driver . '_' . $type;
        $vclass = 'Horde_Image_Effect_' . $type;
        
        if (class_exists($class)) {
            $effect = new $class($params);
        } elseif (class_exists($vclass)) {
            $effect = new $vclass($params);
        } else {
            $message = sprintf(
                'Horde_Image Effect "%s" for "%s" driver not found.',
                $type,
                $driver
            );
            if (!empty($params['logger'])) {
                $params['logger']->err($message);
            }
            throw new Horde_Image_Exception($message);
        }

        if (!empty($params['logger'])) {
            $effect->setLogger($params['logger']);
        }

        return $effect;
    }

}